import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Scale, User, FileText, CheckCircle, Clock, Award, TrendingUp, Shield } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './HistoryPage.css'

const HistoryPage = () => {
  const [consultations, setConsultations] = useState([])

  useEffect(() => {
    loadConsultations()
    const interval = setInterval(loadConsultations, 500)
    return () => clearInterval(interval)
  }, [])

  const loadConsultations = () => {
    const saved = localStorage.getItem('lawConsultations')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setConsultations(parsed)
      } catch (e) {
        setConsultations([])
      }
    } else {
      const defaultConsultations = [
        {
          id: 1,
          topic: 'Семейное право',
          lawyer: 'Анна Соколова',
          status: 'Завершено',
          review: 'Отличная консультация, все разъяснили подробно'
        },
        {
          id: 2,
          topic: 'Трудовое право',
          lawyer: 'Михаил Петров',
          status: 'В процессе',
          review: null
        },
        {
          id: 3,
          topic: 'Жилищное право',
          lawyer: 'Елена Волкова',
          status: 'Завершено',
          review: 'Профессионал своего дела, рекомендую!'
        }
      ]
      setConsultations(defaultConsultations)
      localStorage.setItem('lawConsultations', JSON.stringify(defaultConsultations))
    }
  }

  const getAreaColor = (area) => {
    const colors = {
      'Семейное право': '#1E40AF',
      'Трудовое право': '#3B82F6',
      'Жилищное право': '#60A5FA',
      'Уголовное право': '#2563EB'
    }
    return colors[area] || '#1E40AF'
  }

  const getStatusColor = (status) => {
    return status === 'Завершено' ? '#10B981' : '#F59E0B'
  }

  const totalConsultations = consultations.length
  const completedConsultations = consultations.filter(c => c.status === 'Завершено').length
  const uniqueLawyers = new Set(consultations.map(c => c.lawyer)).size

  return (
    <div className="history-page">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="hero-icon-wrapper"
            initial={{ scale: 0, rotate: -180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Scale size={88} strokeWidth={2.5} className="hero-icon" />
          </motion.div>
          <h1 className="hero-title">История обращений</h1>
          <p className="hero-description">
            Все ваши юридические консультации и обращения в одном месте
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.08, rotate: 3 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(30, 64, 175, 0.18)' }}>
              <FileText size={40} strokeWidth={2.5} color="#1E40AF" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalConsultations}</div>
              <div className="stat-label">Обращений</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.08, rotate: -3 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(16, 185, 129, 0.18)' }}>
              <CheckCircle size={40} strokeWidth={2.5} color="#10B981" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{completedConsultations}</div>
              <div className="stat-label">Завершено</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.08, rotate: 3 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(59, 130, 246, 0.18)' }}>
              <User size={40} strokeWidth={2.5} color="#3B82F6" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{uniqueLawyers}</div>
              <div className="stat-label">Юристов</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/law-1.jpg',
          '/images/law-2.jpg',
          '/images/law-3.jpg',
          '/images/law-4.jpg'
        ]} />

        <motion.section 
          className="consultations-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Ваши обращения</h2>
          {consultations.length > 0 ? (
            <div className="consultations-grid">
              {consultations.map((consultation, index) => (
                <motion.div 
                  key={consultation.id} 
                  className="consultation-card"
                  initial={{ opacity: 0, scale: 0.9, rotate: -5 }}
                  animate={{ opacity: 1, scale: 1, rotate: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.5 }}
                  whileHover={{ 
                    scale: 1.08, 
                    rotate: 4,
                    boxShadow: `0 24px 48px ${getAreaColor(consultation.topic)}35`
                  }}
                  style={{ 
                    borderLeft: `5px solid ${getAreaColor(consultation.topic)}`,
                    background: `linear-gradient(135deg, rgba(255, 255, 255, 0.99) 0%, rgba(239, 246, 255, 0.97) 100%)`
                  }}
                >
                  <div className="consultation-header">
                    <div className="consultation-topic-badge" style={{ background: getAreaColor(consultation.topic) + '20', color: getAreaColor(consultation.topic) }}>
                      {consultation.topic}
                    </div>
                    <div className="consultation-status" style={{ background: getStatusColor(consultation.status) + '20', color: getStatusColor(consultation.status) }}>
                      {consultation.status}
                    </div>
                  </div>
                  <div className="consultation-content">
                    <div className="consultation-info-item">
                      <User size={22} strokeWidth={2.5} color={getAreaColor(consultation.topic)} />
                      <span className="consultation-info-text">{consultation.lawyer}</span>
                    </div>
                    {consultation.review && (
                      <div className="consultation-review">
                        <FileText size={20} strokeWidth={2} color={getAreaColor(consultation.topic)} />
                        <span className="review-text">{consultation.review}</span>
                      </div>
                    )}
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <div className="empty-icon-circle">
                <Scale size={72} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title">Нет обращений</h3>
              <p className="empty-text">Запишитесь на консультацию прямо сейчас</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="benefits-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <div className="benefits-card">
            <h2 className="benefits-title">Почему выбирают нас</h2>
            <div className="benefits-grid">
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(30, 64, 175, 0.15)' }}>
                  <Shield size={36} strokeWidth={2.5} color="#1E40AF" />
                </div>
                <h3 className="benefit-title">Защита прав</h3>
                <p className="benefit-text">Максимальная защита ваших интересов</p>
              </div>
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(16, 185, 129, 0.15)' }}>
                  <Award size={36} strokeWidth={2.5} color="#10B981" />
                </div>
                <h3 className="benefit-title">Опытные юристы</h3>
                <p className="benefit-text">Специалисты с многолетним опытом</p>
              </div>
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(59, 130, 246, 0.15)' }}>
                  <Clock size={36} strokeWidth={2.5} color="#3B82F6" />
                </div>
                <h3 className="benefit-title">Быстрая помощь</h3>
                <p className="benefit-text">Консультации в удобное для вас время</p>
              </div>
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(245, 158, 11, 0.15)' }}>
                  <TrendingUp size={36} strokeWidth={2.5} color="#F59E0B" />
                </div>
                <h3 className="benefit-title">Высокий рейтинг</h3>
                <p className="benefit-text">98% довольных клиентов</p>
              </div>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default HistoryPage

